use std::ops::*;
use std::hash::Hasher;

pub const CONTRACT_ADDRESS_WIDTH: usize = 21;

combi_struct!{ ContractAddress,
    addr: Address
}

impl std::hash::Hash for ContractAddress {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.addr.hash(state);
    }
}


impl Deref for ContractAddress {
    type Target = Address;
    fn deref(&self) -> &Address {
        &self.addr
    }
}


impl ContractAddress {

    // https://en.bitcoin.it/wiki/List_of_address_prefixes
	pub fn calculate(addr: &Address, nonce: &Uint4) -> Self {
		let dts = vec![addr.serialize(), nonce.serialize()].concat();
		let hx32 = x16rs::sha2(dts);
		let hx20 = x16rs::ripemd160(hx32);
		let addr = vec![vec![Address::CONTRACT], hx20.to_vec()].concat();
		ContractAddress::must(addr.try_into().unwrap())
	}

    pub fn must(bts: [u8; CONTRACT_ADDRESS_WIDTH]) -> Self {
        Self::from_addr(Address::from(bts)).unwrap()
    }

    pub fn from_addr(addr: Address) -> Ret<Self> {
        let av = addr.version();
        if av !=  Address::CONTRACT {
            return errf!("address version {} is not CONTRACT", av)
        }
        Ok(Self{addr})
    }

    pub fn to_addr(&self) -> Address {
        self.addr
    }

    pub fn parse(dts: &[u8]) -> Ret<Self> {
        if dts.len() != Address::SIZE {
            return errf!("contract address length error")
        }
        let cadr = Address::from(dts.try_into().unwrap());
        ContractAddress::from_addr(cadr)
    } 


    

}





