

// error define
#[repr(u8)]
#[derive(Default, PartialEq, Debug, Clone, Copy)]
pub enum ItrErrCode {
    ContractError    = 1u8,
    NotFindContract  = 2,
    SystemTypeError  = 3,
    CodeTypeError    = 4,
    InheritsError    = 5,
    LibrarysError    = 6,
    ComplieError     = 7, 
    ContractAddrErr  = 8,
    ContractUpgradeErr = 9,

    CodeError       = 11,
    CodeTooLong     = 12, // code length
    CodeOverflow    = 13, // pc out of limit
    CodeEmpty       = 14,
    IRNodeOverDepth = 15,
    
    InstInvalid    = 21, // 
    InstDisabled   = 22, // 
    ExtActDisabled = 23, // 
    InstNeverTouch = 24, // 
    InstParamsErr  = 25, // 
    
    OutOfGas          = 31,
    OutOfStack        = 32,
    OutOfLocal        = 33,
    OutOfHeap         = 34,
    OutOfMemory       = 35,
    OutOfGlobal       = 36,
    OutOfCallDepth    = 37,
    OutOfLoadContract = 38,
    OutOfValueSize    = 39,
    
    GasError     = 41,
    StackError   = 42,
    LocalError   = 43,
    HeapError    = 44,
    MemoryError  = 45,
    GlobalError  = 46,
    StorageError = 47,
    
    CallNotExist     = 51,
    CallLibOverflow  = 52,
    CallInvalid      = 53,
    CallExitInvalid  = 54,
    CallInCodeCall   = 55,
    CallInSystem     = 56,
    CallOtherInMain  = 57,
    CallNoReturn     = 58,
    CallNotPublic    = 59,
    
    CastFail      = 61,
    CastParamFail = 62,
    
    Arithmetic        = 71,
    BytesHandle       = 72,
    NativeCallError   = 73,

    ExtActCallError  = 81,

    StorageKeyInvalid  = 91,
    StorageKeyNotFind  = 92,
    StorageExpired     = 93,
    StorageNotExpired  = 94,
    StoragePeriodErr   = 95,
    StorageValSizeErr  = 96,
    StorageRestoreNotMatch  = 97,


    ThrowAbort = 101, // user code call

    #[default] NeverError = 255,
}

#[derive(Debug)]
pub struct ItrErr(pub ItrErrCode, pub String);

impl ToString for ItrErr {
    fn to_string(&self) -> String {
        format!("{:?}({}): {}", self.0, self.0 as u8, self.1)
    }
}


impl ItrErr {
    pub fn new(n: ItrErrCode, tip: &str) -> ItrErr {
        ItrErr(n, tip.to_string())
    }
    pub fn code(n: ItrErrCode) -> ItrErr {
        ItrErr(n, "".to_string())
    }
}

// VM Runtime Error
pub type VmrtRes<T> = Result<T, ItrErr>;
pub type VmrtErr = Result<(), ItrErr>;

pub trait IntoVmrt {
    fn into_vmrt(self) -> VmrtRes<Vec<u8>>;
}

impl IntoVmrt for Vec<u8> {
    fn into_vmrt(self) -> Result<Vec<u8>, ItrErr> {
        Ok(self)
    }
}



#[allow(unused)]
macro_rules! map_itr_err {
    ($e : expr) => {
        $e.map_err(|a|a.to_string())
    }
}


#[allow(unused)]
macro_rules! itr_err {
    ($code: expr, $tip: expr) => {
        Err(ItrErr($code, $tip.to_string()))
    }
}

#[allow(unused)]
macro_rules! itr_err_code {
    ($code: expr) => {
        Err(ItrErr($code, "".to_string()))
    }
}

#[allow(unused)]
macro_rules! itr_err_fmt {
    ($code: expr, $( $v: expr),+ ) => {
        Err(ItrErr::new($code, &format!($( $v ),+)))
    }
}
